<?php
/**
 * Plugin Name: As postagens mais acessadas por Léo Vilhena
 * Plugin URI: https://designer.redegni.com.br/plugin-leovilhena-posts-gratuito/
 * Description: Exibe as 3 postagens mais acessadas das últimas 24 horas do seu site - Versão Otimizada
 * Version: 2.0.2
 * Author: Léo Vilhena
 * Author URI: https://designer.redegni.com.br/quem-somos/
 * License: GPL v2 or later
 * Text Domain: leovilhena-posts
 */

// Prevenir acesso direto
if (!defined('ABSPATH')) {
    exit;
}

// Namespace para evitar conflitos
if (!class_exists('LeoVilhena_Posts_Optimized')) {

class LeoVilhena_Posts_Optimized {
    
    private static $instance = null;
    private $option_name = 'leovilhena_posts_settings_v2';
    private $meta_key = 'leovilhena_post_views_v2';
    private $cache_group = 'leovilhena_posts';
    private $cache_time = 300; // 5 minutos de cache
    
    /**
     * Singleton pattern para evitar múltiplas instâncias
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Só carregar hooks do admin quando necessário
        if (is_admin()) {
            $this->init_admin_hooks();
        }
        
        // Hooks do frontend
        $this->init_frontend_hooks();
    }
    
    /**
     * Inicializar hooks do admin
     */
    private function init_admin_hooks() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_filter('manage_posts_columns', array($this, 'add_views_column'));
        add_action('manage_posts_custom_column', array($this, 'show_views_column'), 10, 2);
        add_filter('manage_edit-post_sortable_columns', array($this, 'make_views_column_sortable'));
    }
    
    /**
     * Inicializar hooks do frontend
     */
    private function init_frontend_hooks() {
        // Usar AJAX por padrão para não bloquear o carregamento da página
        add_action('wp_footer', array($this, 'track_post_views_ajax'), 999);
        
        // AJAX handlers
        add_action('wp_ajax_leovilhena_track_view', array($this, 'ajax_track_view'));
        add_action('wp_ajax_nopriv_leovilhena_track_view', array($this, 'ajax_track_view'));
        
        // Registrar shortcode
        add_shortcode('leovilhena_posts', array($this, 'display_top_posts'));
        
        // Registrar widget
        add_action('widgets_init', array($this, 'register_widget'));
        
        // Carregar estilos apenas quando necessário
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        
        // Limpar cache quando post é atualizado
        add_action('save_post', array($this, 'clear_cache'));
    }
    
    /**
     * Adicionar menu no painel administrativo
     */
    public function add_admin_menu() {
        add_options_page(
            'LeoVilhena Posts Settings',
            'LeoVilhena Posts',
            'manage_options',
            'leovilhena-posts',
            array($this, 'settings_page')
        );
        
        add_menu_page(
            'Posts Mais Acessados',
            'Top Posts',
            'manage_options',
            'leovilhena-top-posts',
            array($this, 'stats_page'),
            'dashicons-chart-bar',
            30
        );
    }
    
    /**
     * Registrar configurações
     */
    public function register_settings() {
        register_setting($this->option_name, $this->option_name);
        
        add_settings_section(
            'leovilhena_main_section',
            'Configurações de Exibição',
            array($this, 'settings_section_callback'),
            'leovilhena-posts'
        );
        
        add_settings_field(
            'exclude_admin',
            'Excluir Administradores',
            array($this, 'exclude_admin_field_callback'),
            'leovilhena-posts',
            'leovilhena_main_section'
        );
        
        add_settings_field(
            'show_thumbnail',
            'Mostrar Imagem Destacada',
            array($this, 'show_thumbnail_field_callback'),
            'leovilhena-posts',
            'leovilhena_main_section'
        );
        
        add_settings_field(
            'show_views_count',
            'Mostrar Número de Visualizações',
            array($this, 'show_views_count_field_callback'),
            'leovilhena-posts',
            'leovilhena_main_section'
        );
        
        add_settings_field(
            'show_date',
            'Mostrar Data de Publicação',
            array($this, 'show_date_field_callback'),
            'leovilhena-posts',
            'leovilhena_main_section'
        );
        
        add_settings_field(
            'time_period',
            'Período de Posts',
            array($this, 'time_period_field_callback'),
            'leovilhena-posts',
            'leovilhena_main_section'
        );
        
        add_settings_field(
            'cache_time',
            'Tempo de Cache',
            array($this, 'cache_time_field_callback'),
            'leovilhena-posts',
            'leovilhena_main_section'
        );
    }
    
    public function settings_section_callback() {
        echo '<p>Configure como as postagens mais acessadas serão exibidas.</p>';
    }
    
    public function exclude_admin_field_callback() {
        $options = get_option($this->option_name);
        $value = isset($options['exclude_admin']) ? $options['exclude_admin'] : 1;
        ?>
        <input type="checkbox" name="<?php echo $this->option_name; ?>[exclude_admin]" value="1" <?php checked($value, 1); ?>>
        <label>Não contar visualizações de administradores logados</label>
        <?php
    }
    
    public function show_thumbnail_field_callback() {
        $options = get_option($this->option_name);
        $value = isset($options['show_thumbnail']) ? $options['show_thumbnail'] : 1;
        ?>
        <input type="checkbox" name="<?php echo $this->option_name; ?>[show_thumbnail]" value="1" <?php checked($value, 1); ?>>
        <label>Exibir imagem destacada do post</label>
        <?php
    }
    
    public function show_views_count_field_callback() {
        $options = get_option($this->option_name);
        $value = isset($options['show_views_count']) ? $options['show_views_count'] : 0;
        ?>
        <input type="checkbox" name="<?php echo $this->option_name; ?>[show_views_count]" value="1" <?php checked($value, 1); ?>>
        <label>Mostrar quantidade de visualizações</label>
        <?php
    }
    
    public function show_date_field_callback() {
        $options = get_option($this->option_name);
        $value = isset($options['show_date']) ? $options['show_date'] : 1;
        ?>
        <input type="checkbox" name="<?php echo $this->option_name; ?>[show_date]" value="1" <?php checked($value, 1); ?>>
        <label>Mostrar data de publicação</label>
        <?php
    }
    
    public function time_period_field_callback() {
        $options = get_option($this->option_name);
        $value = isset($options['time_period']) ? $options['time_period'] : '24hours';
        ?>
        <select name="<?php echo $this->option_name; ?>[time_period]">
            <option value="all" <?php selected($value, 'all'); ?>>Todos os tempos</option>
            <option value="24hours" <?php selected($value, '24hours'); ?>>Últimas 24 horas</option>
            <option value="7days" <?php selected($value, '7days'); ?>>Últimos 7 dias</option>
            <option value="30days" <?php selected($value, '30days'); ?>>Últimos 30 dias</option>
        </select>
        <p class="description">Mostrar posts mais acessados deste período (padrão: 24 horas)</p>
        <?php
    }
    
    public function cache_time_field_callback() {
        $options = get_option($this->option_name);
        $value = isset($options['cache_time']) ? $options['cache_time'] : 300;
        ?>
        <select name="<?php echo $this->option_name; ?>[cache_time]">
            <option value="60" <?php selected($value, 60); ?>>1 minuto</option>
            <option value="300" <?php selected($value, 300); ?>>5 minutos (Recomendado)</option>
            <option value="600" <?php selected($value, 600); ?>>10 minutos</option>
            <option value="1800" <?php selected($value, 1800); ?>>30 minutos</option>
            <option value="3600" <?php selected($value, 3600); ?>>1 hora</option>
        </select>
        <p class="description">Quanto tempo manter os dados em cache (melhora performance)</p>
        <?php
    }
    
    /**
     * Página de configurações
     */
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1>LeoVilhena Posts - Configurações</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields($this->option_name);
                do_settings_sections('leovilhena-posts');
                submit_button();
                ?>
            </form>
            
            <hr>
            
            <h2>Como Usar</h2>
            <p><strong>Shortcode:</strong> Use <code>[leovilhena_posts]</code> em qualquer página ou post.</p>
            <p><strong>Widget:</strong> Vá em Aparência > Widgets e adicione o widget "LeoVilhena Posts".</p>
            <p><strong>Template PHP:</strong> Use <code>&lt;?php echo do_shortcode('[leovilhena_posts]'); ?&gt;</code></p>
            
            <h3>Opções do Shortcode</h3>
            <ul>
                <li><code>[leovilhena_posts limit="3"]</code> - Número de posts (padrão: 3)</li>
                <li><code>[leovilhena_posts title="Mais Lidos"]</code> - Título personalizado</li>
                <li><code>[leovilhena_posts layout="grid"]</code> - Layout em grid (padrão: list)</li>
                <li><code>[leovilhena_posts category="5"]</code> - Filtrar por categoria (ID)</li>
            </ul>
            
            <h3>Otimizações</h3>
            <p>✅ Sistema de cache implementado (reduz queries no banco)</p>
            <p>✅ Rastreamento via AJAX (não bloqueia carregamento da página)</p>
            <p>✅ Código encapsulado (evita conflitos com outros plugins)</p>
            
            <h3>Gerenciar Cache</h3>
            <p>
                <a href="<?php echo admin_url('admin.php?page=leovilhena-posts&action=clear_cache'); ?>" 
                   class="button button-secondary">
                    Limpar Cache Agora
                </a>
            </p>
            
            <h3>Reiniciar Contadores</h3>
            <p>
                <a href="<?php echo admin_url('admin.php?page=leovilhena-top-posts&action=reset'); ?>" 
                   class="button button-secondary" 
                   onclick="return confirm('Tem certeza? Isso irá resetar todas as contagens de visualizações!');">
                    Resetar Todas as Visualizações
                </a>
            </p>
        </div>
        <?php
        
        // Processar ação de limpar cache
        if (isset($_GET['action']) && $_GET['action'] === 'clear_cache' && current_user_can('manage_options')) {
            $this->clear_cache();
            echo '<div class="notice notice-success"><p>Cache limpo com sucesso!</p></div>';
        }
    }
    
    /**
     * Página de estatísticas
     */
    public function stats_page() {
        if (isset($_GET['action']) && $_GET['action'] === 'reset' && current_user_can('manage_options')) {
            $this->reset_all_views();
            echo '<div class="notice notice-success"><p>Todas as visualizações foram resetadas!</p></div>';
        }
        
        $top_posts = $this->get_top_posts(20);
        ?>
        <div class="wrap">
            <h1>📊 Posts Mais Acessados de Todos os Tempos</h1>
            
            <div class="leovilhena-stats-summary">
                <div class="stat-box">
                    <h3><?php echo $this->get_total_views(); ?></h3>
                    <p>Total de Visualizações</p>
                </div>
                <div class="stat-box">
                    <h3><?php echo count($top_posts); ?></h3>
                    <p>Posts Rastreados</p>
                </div>
                <div class="stat-box">
                    <h3><?php echo $top_posts ? number_format($top_posts[0]['views'], 0, ',', '.') : 0; ?></h3>
                    <p>Post Mais Visto</p>
                </div>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th width="50">Posição</th>
                        <th>Título do Post</th>
                        <th width="150">Visualizações</th>
                        <th width="150">Data de Publicação</th>
                        <th width="100">Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($top_posts): ?>
                        <?php foreach ($top_posts as $index => $post_data): ?>
                            <tr>
                                <td><strong><?php echo ($index + 1); ?>º</strong></td>
                                <td>
                                    <a href="<?php echo get_permalink($post_data['id']); ?>" target="_blank">
                                        <?php echo esc_html($post_data['title']); ?>
                                    </a>
                                </td>
                                <td>
                                    <strong><?php echo number_format($post_data['views'], 0, ',', '.'); ?></strong>
                                    <span class="dashicons dashicons-visibility"></span>
                                </td>
                                <td><?php echo get_the_date('d/m/Y', $post_data['id']); ?></td>
                                <td>
                                    <a href="<?php echo get_edit_post_link($post_data['id']); ?>" class="button button-small">
                                        Editar
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="5">Nenhum post com visualizações ainda.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <style>
                .leovilhena-stats-summary {
                    display: grid;
                    grid-template-columns: repeat(3, 1fr);
                    gap: 20px;
                    margin: 30px 0;
                }
                .stat-box {
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    padding: 30px;
                    border-radius: 10px;
                    text-align: center;
                }
                .stat-box h3 {
                    font-size: 36px;
                    margin: 0 0 10px 0;
                    color: white;
                }
                .stat-box p {
                    margin: 0;
                    opacity: 0.9;
                }
            </style>
        </div>
        <?php
    }
    
    /**
     * Rastrear visualizações via AJAX (não bloqueia o carregamento)
     */
    public function track_post_views_ajax() {
        if (!is_single()) return;
        
        $options = get_option($this->option_name);
        $exclude_admin = isset($options['exclude_admin']) ? $options['exclude_admin'] : 1;
        
        if ($exclude_admin && current_user_can('manage_options')) {
            return;
        }
        
        $post_id = get_the_ID();
        ?>
        <script>
        (function() {
            if ('requestIdleCallback' in window) {
                requestIdleCallback(function() {
                    trackView();
                });
            } else {
                setTimeout(trackView, 2000);
            }
            
            function trackView() {
                var xhr = new XMLHttpRequest();
                xhr.open('POST', '<?php echo admin_url('admin-ajax.php'); ?>', true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.send('action=leovilhena_track_view&post_id=<?php echo $post_id; ?>');
            }
        })();
        </script>
        <?php
    }
    
    /**
     * AJAX handler otimizado
     */
    public function ajax_track_view() {
        if (!isset($_POST['post_id'])) {
            wp_die();
        }
        
        $post_id = intval($_POST['post_id']);
        
        $options = get_option($this->option_name);
        $exclude_admin = isset($options['exclude_admin']) ? $options['exclude_admin'] : 1;
        
        if (!($exclude_admin && current_user_can('manage_options'))) {
            $this->increment_post_views($post_id);
        }
        
        wp_die();
    }
    
    /**
     * Incrementar visualizações de forma otimizada
     */
    private function increment_post_views($post_id) {
        $count = get_post_meta($post_id, $this->meta_key, true);
        $count = $count ? intval($count) : 0;
        $count++;
        update_post_meta($post_id, $this->meta_key, $count);
        
        // Limpar cache deste post específico
        wp_cache_delete('top_posts_' . $post_id, $this->cache_group);
    }
    
    /**
     * Obter posts com sistema de cache
     */
    private function get_top_posts($limit = 3, $category = null) {
        $options = get_option($this->option_name);
        $time_period = isset($options['time_period']) ? $options['time_period'] : '24hours';
        $cache_time = isset($options['cache_time']) ? intval($options['cache_time']) : 300;
        
        // Criar chave única de cache
        $cache_key = 'top_posts_' . $limit . '_' . $time_period . '_' . ($category ? $category : 'all');
        
        // Tentar buscar do cache
        $cached_posts = wp_cache_get($cache_key, $this->cache_group);
        if (false !== $cached_posts) {
            return $cached_posts;
        }
        
        // Se não houver cache, buscar do banco
        $args = array(
            'post_type' => 'post',
            'posts_per_page' => $limit * 2, // Buscar o dobro para ter margem
            'post_status' => 'publish',
            'meta_key' => $this->meta_key,
            'orderby' => 'meta_value_num',
            'order' => 'DESC',
            'no_found_rows' => true, // Otimização: não contar total
            'update_post_meta_cache' => false, // Otimização
            'update_post_term_cache' => false, // Otimização
        );
        
        if ($time_period !== 'all') {
            $date_query = array();
            
            switch ($time_period) {
                case '24hours':
                    $date_query = array('after' => '24 hours ago');
                    break;
                case '7days':
                    $date_query = array('after' => '7 days ago');
                    break;
                case '30days':
                    $date_query = array('after' => '30 days ago');
                    break;
            }
            
            if (!empty($date_query)) {
                $args['date_query'] = array($date_query);
            }
        }
        
        if ($category) {
            $args['cat'] = $category;
        }
        
        $query = new WP_Query($args);
        $posts = array();
        
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $views = get_post_meta(get_the_ID(), $this->meta_key, true);
                
                if ($views) {
                    $posts[] = array(
                        'id' => get_the_ID(),
                        'title' => get_the_title(),
                        'url' => get_permalink(),
                        'views' => intval($views),
                        'thumbnail' => get_the_post_thumbnail_url(get_the_ID(), 'medium'),
                        'date' => get_the_date('d/m/Y'),
                    );
                }
            }
            wp_reset_postdata();
        }
        
        $posts = array_slice($posts, 0, $limit);
        
        // Salvar no cache
        wp_cache_set($cache_key, $posts, $this->cache_group, $cache_time);
        
        return $posts;
    }
    
    /**
     * Obter total de visualizações (com cache)
     */
    private function get_total_views() {
        $cache_key = 'total_views';
        $cached_total = wp_cache_get($cache_key, $this->cache_group);
        
        if (false !== $cached_total) {
            return $cached_total;
        }
        
        global $wpdb;
        $total = $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(meta_value) FROM {$wpdb->postmeta} WHERE meta_key = %s",
            $this->meta_key
        ));
        
        $formatted = number_format($total ? $total : 0, 0, ',', '.');
        
        wp_cache_set($cache_key, $formatted, $this->cache_group, 600);
        
        return $formatted;
    }
    
    /**
     * Limpar cache
     */
    public function clear_cache() {
        wp_cache_flush();
    }
    
    /**
     * Resetar visualizações
     */
    private function reset_all_views() {
        global $wpdb;
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
            $this->meta_key
        ));
        $this->clear_cache();
    }
    
    /**
     * Shortcode otimizado
     */
    public function display_top_posts($atts) {
        $atts = shortcode_atts(array(
            'limit' => 3,
            'title' => 'Posts Mais Acessados',
            'layout' => 'list',
            'category' => null,
        ), $atts);
        
        $options = get_option($this->option_name);
        $show_thumbnail = isset($options['show_thumbnail']) ? $options['show_thumbnail'] : 1;
        $show_views = isset($options['show_views_count']) ? $options['show_views_count'] : 0;
        $show_date = isset($options['show_date']) ? $options['show_date'] : 1;
        $time_period = isset($options['time_period']) ? $options['time_period'] : '24hours';
        
        $posts = $this->get_top_posts($atts['limit'], $atts['category']);
        
        if (empty($posts)) {
            return '<p>Nenhum post encontrado neste período.</p>';
        }
        
        ob_start();
        ?>
        <div class="leovilhena-top-posts <?php echo esc_attr($atts['layout']); ?>-layout">
            <?php if ($atts['title']): ?>
                <h3 class="lv-posts-title">
                    <?php echo esc_html($atts['title']); ?>
                    <?php if ($time_period !== 'all'): ?>
                        <span class="lv-period-badge">
                            <?php 
                            switch($time_period) {
                                case '24hours': echo '24h'; break;
                                case '7days': echo '7 dias'; break;
                                case '30days': echo '30 dias'; break;
                            }
                            ?>
                        </span>
                    <?php endif; ?>
                </h3>
            <?php endif; ?>
            
            <div class="lv-posts-container">
                <?php foreach ($posts as $index => $post): ?>
                    <div class="lv-post-item">
                        <div class="lv-post-rank"><?php echo ($index + 1); ?>º</div>
                        
                        <?php if ($show_thumbnail && $post['thumbnail']): ?>
                            <div class="lv-post-thumbnail">
                                <a href="<?php echo esc_url($post['url']); ?>">
                                    <img src="<?php echo esc_url($post['thumbnail']); ?>" 
                                         alt="<?php echo esc_attr($post['title']); ?>"
                                         loading="lazy">
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <div class="lv-post-content">
                            <h4 class="lv-post-title">
                                <a href="<?php echo esc_url($post['url']); ?>">
                                    <?php echo esc_html($post['title']); ?>
                                </a>
                            </h4>
                            
                            <div class="lv-post-meta">
                                <?php if ($show_views): ?>
                                    <span class="lv-views">
                                        <span class="dashicons dashicons-visibility"></span>
                                        <?php echo number_format($post['views'], 0, ',', '.'); ?> visualizações
                                    </span>
                                <?php endif; ?>
                                
                                <?php if ($show_date): ?>
                                    <span class="lv-date">
                                        <span class="dashicons dashicons-calendar-alt"></span>
                                        <?php echo $post['date']; ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Colunas do admin
     */
    public function add_views_column($columns) {
        $columns['post_views'] = 'Visualizações';
        return $columns;
    }
    
    public function show_views_column($column_name, $post_id) {
        if ($column_name === 'post_views') {
            $views = get_post_meta($post_id, $this->meta_key, true);
            echo $views ? number_format($views, 0, ',', '.') : '0';
        }
    }
    
    public function make_views_column_sortable($columns) {
        $columns['post_views'] = 'post_views';
        return $columns;
    }
    
    /**
     * Registrar widget
     */
    public function register_widget() {
        register_widget('LeoVilhena_Posts_Widget_V2');
    }
    
    /**
     * Enfileirar estilos apenas quando necessário
     */
    public function enqueue_styles() {
        // Só carregar se o shortcode está sendo usado
        if (!is_admin() && (has_shortcode(get_post()->post_content, 'leovilhena_posts') || is_active_widget(false, false, 'leovilhena_posts_widget_v2'))) {
            wp_enqueue_style(
                'leovilhena-posts-style',
                plugins_url('css/style.css', __FILE__),
                array(),
                '2.0.2'
            );
        }
    }
}

/**
 * Widget otimizado
 */
class LeoVilhena_Posts_Widget_V2 extends WP_Widget {
    
    public function __construct() {
        parent::__construct(
            'leovilhena_posts_widget_v2',
            'LeoVilhena Posts (Otimizado)',
            array('description' => 'Exibe os posts mais acessados - Versão Otimizada')
        );
    }
    
    public function widget($args, $instance) {
        echo $args['before_widget'];
        
        $title = !empty($instance['title']) ? $instance['title'] : 'Posts Mais Acessados';
        $limit = !empty($instance['limit']) ? $instance['limit'] : 3;
        $layout = !empty($instance['layout']) ? $instance['layout'] : 'list';
        
        echo do_shortcode("[leovilhena_posts title='" . esc_attr($title) . "' limit='" . esc_attr($limit) . "' layout='" . esc_attr($layout) . "']");
        
        echo $args['after_widget'];
    }
    
    public function form($instance) {
        $title = isset($instance['title']) ? $instance['title'] : 'Posts Mais Acessados';
        $limit = isset($instance['limit']) ? $instance['limit'] : 3;
        $layout = isset($instance['layout']) ? $instance['layout'] : 'list';
        ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>">Título:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('title'); ?>" 
                   name="<?php echo $this->get_field_name('title'); ?>" type="text" 
                   value="<?php echo esc_attr($title); ?>">
        </p>
        <p>
            <label for="<?php echo $this->get_field_id('limit'); ?>">Número de Posts:</label>
            <input class="widefat" id="<?php echo $this->get_field_id('limit'); ?>" 
                   name="<?php echo $this->get_field_name('limit'); ?>" type="number" 
                   value="<?php echo esc_attr($limit); ?>" min="1" max="10">
        </p>
        <p>
            <label for="<?php echo $this->get_field_id('layout'); ?>">Layout:</label>
            <select class="widefat" id="<?php echo $this->get_field_id('layout'); ?>" 
                    name="<?php echo $this->get_field_name('layout'); ?>">
                <option value="list" <?php selected($layout, 'list'); ?>>Lista</option>
                <option value="grid" <?php selected($layout, 'grid'); ?>>Grid</option>
            </select>
        </p>
        <?php
    }
    
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = !empty($new_instance['title']) ? sanitize_text_field($new_instance['title']) : '';
        $instance['limit'] = !empty($new_instance['limit']) ? intval($new_instance['limit']) : 3;
        $instance['layout'] = !empty($new_instance['layout']) ? sanitize_text_field($new_instance['layout']) : 'list';
        return $instance;
    }
}

// Inicializar usando Singleton
function leovilhena_posts_optimized_init() {
    return LeoVilhena_Posts_Optimized::get_instance();
}

// Hook de inicialização com prioridade baixa
add_action('plugins_loaded', 'leovilhena_posts_optimized_init', 20);

} // Fim do if !class_exists
